
    // ===== Dynamic Header: Clock =====
    const clock = document.querySelector('#clock span');
    function pad(n) { return (n < 10 ? '0' : '') + n }
    function tick() {
      const d = new Date();
      const h = pad(d.getHours());
      const m = pad(d.getMinutes());
      clock.textContent = `${h}:${m}`;
    }
    setInterval(tick, 1000); tick();

    // ===== Sidebar: dropdowns =====
    const dropdownTriggers = [...document.querySelectorAll('[data-toggle="submenu"]')];
    dropdownTriggers.forEach(btn => {
      const item = btn.closest('.nav-item');
      const pane = item.querySelector('.submenu');
      btn.addEventListener('click', () => {
        const expanded = btn.getAttribute('aria-expanded') === 'true';
        btn.setAttribute('aria-expanded', String(!expanded));
        btn.classList.toggle('active', !expanded);
        pane.style.height = !expanded ? (pane.scrollHeight + 'px') : '0px';
        btn.querySelector('.chev').style.transform = !expanded ? 'rotate(180deg)' : 'rotate(0)';
      });
    });
    function closeSiblings(currentBtn) {
      dropdownTriggers.forEach(btn => {
        if (btn !== currentBtn) {
          const pane = btn.closest('.nav-item').querySelector('.submenu');
          btn.setAttribute('aria-expanded', 'false');
          btn.classList.remove('active');
          pane.style.height = '0px';
          btn.querySelector('.chev').style.transform = 'rotate(0)';
        }
      })
    }
    dropdownTriggers.forEach(btn => btn.addEventListener('click', () => closeSiblings(btn)));

    // ===== Sidebar: collapse & mobile toggle =====
    const app = document.getElementById('app');
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const collapseToggle = document.getElementById('collapseToggle');
    const closeSidebarMobile = document.getElementById('closeSidebarMobile');
    
    sidebarToggle.addEventListener('click', () => {
      if (window.matchMedia('(max-width: 860px)').matches) {
        sidebar.classList.toggle('open');
      } else {
        app.classList.toggle('collapsed');
      }
    });
    
    collapseToggle.addEventListener('click', () => {
      app.classList.toggle('collapsed');
    });
    
    closeSidebarMobile.addEventListener('click', () => {
      sidebar.classList.remove('open');
    });
    
    document.querySelector('.main').addEventListener('click', () => {
      if (window.matchMedia('(max-width: 860px)').matches) {
        sidebar.classList.remove('open');
      }
    });

    // ===== Keyboard shortcut for search focus =====
    const searchInput = document.querySelector('.header .search input');
    window.addEventListener('keydown', (e) => {
      if (e.key === '/' && document.activeElement !== searchInput) {
        e.preventDefault(); searchInput.focus();
      }
    });

    // ===== Set current year in footer =====
    document.getElementById('year').textContent = new Date().getFullYear();

    // ===== Sample Renewal Data =====
    const renewals = [
      {
        id: 'REN-001',
        customerId: 'CUS-001',
        name: 'Ahmed Raza',
        company: 'Tech Solutions Ltd',
        currentPlan: 'Premium Website',
        renewalDate: '2024-04-15',
        daysLeft: 5,
        amount: 'PKR 15,000',
        status: 'pending',
        email: 'ahmed@techsolutions.com'
      },
      {
        id: 'REN-002',
        customerId: 'CUS-002',
        name: 'Sana Khan',
        company: 'Digital Marketing Pro',
        currentPlan: 'Business Software',
        renewalDate: '2024-04-20',
        daysLeft: 10,
        amount: 'PKR 45,000',
        status: 'pending',
        email: 'sana@dmp.com'
      },
      {
        id: 'REN-003',
        customerId: 'CUS-003',
        name: 'Bilal Ahmed',
        company: 'Startup Innovations',
        currentPlan: 'Basic Website',
        renewalDate: '2024-03-01',
        daysLeft: -15,
        amount: 'PKR 120,000',
        status: 'expired',
        email: 'bilal@startup.com'
      },
      {
        id: 'REN-004',
        customerId: 'CUS-004',
        name: 'Hira Malik',
        company: 'E-commerce Store',
        currentPlan: 'E-commerce Package',
        renewalDate: '2024-05-05',
        daysLeft: 25,
        amount: 'PKR 25,000',
        status: 'upcoming',
        email: 'hira@estore.com'
      },
      {
        id: 'REN-005',
        customerId: 'CUS-005',
        name: 'Usman Ali',
        company: 'Consulting Firm',
        currentPlan: 'Enterprise Software',
        renewalDate: '2024-03-10',
        daysLeft: -10,
        amount: 'PKR 60,000',
        status: 'expired',
        email: 'usman@consulting.com'
      },
      {
        id: 'REN-006',
        customerId: 'CUS-006',
        name: 'Fatima Noor',
        company: 'Creative Agency',
        currentPlan: 'Professional Hosting',
        renewalDate: '2024-04-25',
        daysLeft: 15,
        amount: 'PKR 18,000',
        status: 'pending',
        email: 'fatima@creative.com'
      },
      {
        id: 'REN-007',
        customerId: 'CUS-007',
        name: 'Kamran Butt',
        company: 'Manufacturing Corp',
        currentPlan: 'Custom Software',
        renewalDate: '2024-06-20',
        daysLeft: 51,
        amount: 'PKR 200,000',
        status: 'upcoming',
        email: 'kamran@manufacturing.com'
      },
      {
        id: 'REN-008',
        customerId: 'CUS-008',
        name: 'Ayesha Siddiqui',
        company: 'Healthcare Solutions',
        currentPlan: 'Medical Website',
        renewalDate: '2024-04-01',
        daysLeft: -9,
        amount: 'PKR 22,000',
        status: 'expired',
        email: 'ayesha@healthcare.com'
      },
      {
        id: 'REN-009',
        customerId: 'CUS-009',
        name: 'Zain Abid',
        company: 'Real Estate Partners',
        currentPlan: 'Domain & Hosting',
        renewalDate: '2024-05-15',
        daysLeft: 35,
        amount: 'PKR 30,000',
        status: 'upcoming',
        email: 'zain@realestate.com'
      },
      {
        id: 'REN-010',
        customerId: 'CUS-010',
        name: 'Nadia Shah',
        company: 'Education First',
        currentPlan: 'Education Package',
        renewalDate: '2024-04-18',
        daysLeft: 8,
        amount: 'PKR 28,000',
        status: 'pending',
        email: 'nadia@education.com'
      }
    ];

    // ===== Populate Renewals Table =====
    function populateRenewalsTable() {
      const tableBody = document.getElementById('renewalsTable');
      tableBody.innerHTML = '';
      
      renewals.forEach(renewal => {
        const row = document.createElement('tr');
        
        // Get initials for avatar
        const initials = renewal.name.split(' ').map(n => n[0]).join('').toUpperCase();
        
        // Format renewal date
        const renewalDate = new Date(renewal.renewalDate).toLocaleDateString('en-US', {
          year: 'numeric',
          month: 'short',
          day: 'numeric'
        });
        
        // Determine days remaining class
        let daysClass = 'normal';
        if (renewal.daysLeft < 0) {
          daysClass = 'critical';
        } else if (renewal.daysLeft <= 7) {
          daysClass = 'warning';
        }
        
        row.innerHTML = `
          <td>
            <input type="checkbox" class="renewal-checkbox" data-id="${renewal.id}">
          </td>
          <td>
            <div style="display: flex; align-items: center; gap: 12px;">
              <div class="customer-avatar">${initials}</div>
              <div>
                <div style="font-weight: 600;">${renewal.name}</div>
                <div style="font-size: 12px; color: var(--muted);">${renewal.email}</div>
              </div>
            </div>
          </td>
          <td>${renewal.currentPlan}</td>
          <td>${renewalDate}</td>
          <td>
            <span class="days-remaining ${daysClass}">
              ${renewal.daysLeft >= 0 ? renewal.daysLeft + ' days' : 'Expired'}
            </span>
          </td>
          <td>
            <div style="font-weight: 600;">${renewal.amount}</div>
          </td>
          <td>
            <span class="status ${renewal.status}">${renewal.status.charAt(0).toUpperCase() + renewal.status.slice(1)}</span>
          </td>
          <td>
            <div class="action-buttons">
              <button class="action-btn view" title="View Details" data-id="${renewal.id}">
                <i class="fa-solid fa-eye"></i>
              </button>
              <button class="action-btn renew" title="Renew Now" data-id="${renewal.id}">
                <i class="fa-solid fa-rotate-right"></i>
              </button>
              <button class="action-btn remind" title="Send Reminder" data-id="${renewal.id}">
                <i class="fa-solid fa-bell"></i>
              </button>
            </div>
          </td>
        `;
        
        tableBody.appendChild(row);
      });
    }

    // ===== Table Search Functionality =====
    document.getElementById('tableSearch').addEventListener('input', function(e) {
      const searchTerm = e.target.value.toLowerCase();
      const rows = document.querySelectorAll('#renewalsTable tr');
      
      rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(searchTerm) ? '' : 'none';
      });
    });

    // ===== Select All Checkbox =====
    document.getElementById('selectAll').addEventListener('change', function(e) {
      const checkboxes = document.querySelectorAll('.renewal-checkbox');
      checkboxes.forEach(checkbox => {
        checkbox.checked = e.target.checked;
      });
    });

    // ===== Action Button Handlers =====
    document.addEventListener('click', function(e) {
      if (e.target.closest('.action-btn.view')) {
        const renewalId = e.target.closest('.action-btn').getAttribute('data-id');
        alert(`View renewal details for ID: ${renewalId}`);
      }
      
      if (e.target.closest('.action-btn.renew')) {
        const renewalId = e.target.closest('.action-btn').getAttribute('data-id');
        if (confirm('Are you sure you want to renew this subscription?')) {
          alert(`Subscription renewed for ID: ${renewalId}`);
          // In a real application, you would update the renewal status
        }
      }
      
      if (e.target.closest('.action-btn.remind')) {
        const renewalId = e.target.closest('.action-btn').getAttribute('data-id');
        alert(`Reminder sent for renewal ID: ${renewalId}`);
      }
    });

    // ===== Bulk Renew Button =====
    document.getElementById('bulkRenewBtn').addEventListener('click', function() {
      const selectedCheckboxes = document.querySelectorAll('.renewal-checkbox:checked');
      if (selectedCheckboxes.length === 0) {
        alert('Please select at least one renewal to process.');
        return;
      }
      
      const selectedIds = Array.from(selectedCheckboxes).map(cb => cb.getAttribute('data-id'));
      if (confirm(`Are you sure you want to renew ${selectedIds.length} selected subscriptions?`)) {
        alert(`Bulk renewal processed for ${selectedIds.length} subscriptions.`);
        // In a real application, you would process all selected renewals
      }
    });

    // ===== Pagination Controls =====
    document.querySelectorAll('.page-btn').forEach(btn => {
      btn.addEventListener('click', function() {
        if (!this.querySelector('i')) { // Don't change active state for arrow buttons
          document.querySelectorAll('.page-btn').forEach(b => b.classList.remove('active'));
          this.classList.add('active');
        }
      });
    });

    // ===== Initialize the page =====
    document.addEventListener('DOMContentLoaded', function() {
      // Set current page as active in sidebar
      const currentPage = 'customerrenewal.html';
      document.querySelectorAll('.nav a').forEach(link => {
        if (link.getAttribute('href') === currentPage) {
          link.classList.add('active');
        }
      });
      
      // Populate the renewals table
      populateRenewalsTable();
    });
 